/*
 * definitions for dealing with the fabric
 */

#define FMS_DB_FLUSH_DELAY (3*1000)	/* delay between fabric DB flushes */

#define FMS_BOGUS_SWITCH_STR "bogusswitch"

/*
 * data structures
 */

/*
 * FMS global fabric specific variables
 */
struct fms_fabvars {
  struct lf_fabric *fabric;		/* current fabric */
#ifdef CLONE
  struct lf_fabric *db_fabric;		/* fabric in database */
#endif

  struct lf_fabric_db *fabric_db;	/* database handles for fabric */

  struct lf_event *db_flush_task;	/* scheduler event for writing DB */

  int auto_commit;	/* bitfield indicating what changes to auto-commit */
  int bogus_switch_index;		/* next bogus switch index */

  int num_fmas;		/* number of connected FMA */
  int map_version;	/* current map version */

  char *product_info_dir;
};


/*
 * per-fabric struct
 */
struct fms_fabric {

  struct lf_event *push_map_task;	/* for delayed map push */

  struct lf_event *request_map_task;	/* scheduler event for requesting map */
  struct lf_event *request_map_timeout;	/* timeout for map request */

  struct fms_fma_desc *invalid_map_reporter;	/* who said map invalid */
  struct fms_fma_desc *request_map_adp;	/* FMA doing the map */
  lf_string_t last_mapper;		/* hostname of last mapper */

  void *topo_map;	/* pointer to a buffer containing topo_map */
  unsigned int topo_map_size;

  /* xbar resolution */
  int inv_rt_seq_no;
  int resolve_in_prog;
  int inv_rt_xbar_index;
  int inv_rt_host_in_port;
  int inv_rt_retries;
  int num_bogus_xbars;
};
#define FMS_FABRIC(F) ((struct fms_fabric *)(F)->user.fms)

/*
 * per-enclosure struct
 */
struct fms_enclosure {
  struct fms_switch_monitor *monitor;	/* monitor running on this switch */
  struct fms_alert *alerts_anchor;	/* alert chain for this enclosure */

#ifdef CLONE
  struct lf_enclosure *clone_mate;	/* partner in clone fabric */
#endif

  struct lf_switch_data_change *change_list;	/* changes noted by query */

  int eid;			/* a unique ID for this enclosure */

  int is_bogus;			/* if true, this is not a real enclosure */
};
#define FMS_ENC(P) ((struct fms_enclosure *)(P)->user.fms)


/*
 * per-linecard struct
 */
struct fms_linecard {
  struct fms_alert *alerts_anchor;	/* alert chain for this linecard */

  int new;		/* TRUE until first pass through monitoring code */

  int hot;		/* boolean, true is card is hot */
};
#define FMS_LC(P) ((struct fms_linecard *)(P)->user.fms)


/*
 * per-linecard-transceiver struct
 */
struct fms_lc_xcvr {
  struct fms_alert *alerts_anchor;	/* alert chain for this xcvr */
};
#define FMS_LC_XCVR(P) ((struct fms_lc_xcvr *)(P)->user.fms_lc)

/*
 * A structure of stuff for each xbar port
 */
struct fms_xbar_port {
  struct fms_link *link;	/* one link per xbar port */

  /* low-frequency monitoring values */
  struct fms_xbar_lfmon {
    int badcrcs;
  } lfmon;

  /* very-low-frequency monitoring values */
  struct fms_xbar_vlfmon{
    unsigned int portflipcount;
  } vlfmon;

  /*
   * private vars for fabric resolution
   */
  int res_inv_rt_count;		/* saved invalid route count */
  int res_inv_rt_diff;		/* # of invalid rts seen over interval */
};

/*
 * per-xbar struct
 */
struct fms_xbar {
  struct fms_alert *alerts_anchor;	/* alert chain for this xbar */

  struct fms_xbar_port **x_port_data;		/* per-port data */

  /* breadth-first serach bookkeeping */
  unsigned int bf_serial;
  struct lf_xbar *bf_next;

  /* for merging with a reference fabric */
  struct lf_xbar *mrf_mate;

  /* for xbar resoloution */
  int mrf_resolved;
};
#define FMS_XBAR(P) ((struct fms_xbar *)((P)->user.fms))
#define FMS_XBAR_N(N) (FMS_XBAR(LF_XBAR(N)))


/*
 * per-host struct
 */
struct fms_host {
  struct fms_fma_desc *adp;
  int update_state;		/* what kind of updates are pending or not */
  int commit;			/* If true, commit changes to database */

#ifdef CLONE
  struct lf_host *clone_mate;	/* partner in clone fabric */
#endif

  int fma_seen;		/* true if FMA for this host has ever been seen */
  struct fms_alert *alerts_anchor;	/* alert chain for this host */

  int disconnected;	/* TRUE if this host appears to be disconnected */
  int is_bogus;		/* true indicates this is just a placeholder host */
};
#define FMS_HOST(P) ((struct fms_host *)((P)->user.fms))


/*
 * A structure of stuff for each NIC port
 */
struct fms_nic_port {
  struct fms_link *link;	/* one link per NIC port */
};

/*
 * per-nic struct
 */
struct fms_nic {
  struct fms_nic_port **nic_port_data;	/* per-port data */

  int seen;			/* used to reconcile reported NICs */

  struct lf_nic *mrf_mate;	/* mate for merging with reference fabric */
};
#define FMS_NIC(P) ((struct fms_nic *)((P)->user.fms))
#define FMS_NIC_N(N) (FMS_NIC(LF_NIC(N)))


/* a link struct, referred to by either end of the link.  This 
 * is a topological link, not a physical one.
 */
struct fms_link {
  int fl_ref_cnt;		/* reference count */

  struct fms_alert *fl_alerts_anchor;	/* alert chain for this link */
};
#define FMS_LINK_OTHER_NODE(LINK, NODE) \
	((LINK)->node1 == LF_NODE(NODE) ? (LINK)->node2 : (LINK)->node1)

/*
 * prototypes
 */
void fms_fill_host_private(struct lf_host *hp);
void fms_fill_linecard_private(struct lf_linecard *lp);
void fms_fill_nic_private(struct lf_nic *np);
struct lf_host *fms_clone_host(struct lf_host *hp);
struct lf_enclosure *fms_clone_enclosure(struct lf_enclosure *ep);

void fms_free_fabric(struct lf_fabric *fp);
void fms_free_nic_private(struct lf_nic *np);
void fms_free_nic(struct lf_nic *np);
void fms_free_host(struct lf_host *hp);
void fms_free_host_private(struct lf_host *hp);
struct fms_link *fms_get_node_link(lf_node_t *np, int p);
void fms_set_node_link(lf_node_t *np, int p, struct fms_link *lp);
void fms_dereference_link(struct fms_link *linkp);
void fms_remove_enclosure(struct lf_enclosure *ep);
void fms_remove_enclosure_from_fabric(struct lf_fabric *fp,
                                      struct lf_enclosure *ep);
void fms_remove_topo_link(union lf_node *np, int port);
void fms_remove_nic_links(struct lf_nic *nicp);

int fms_monitor_enclosure(struct lf_enclosure *ep);
void fms_fma_init_host_alerts(struct lf_host *hp);
void fms_fma_init_enclosure_alerts(struct lf_enclosure *ep);
void fms_fma_init_linecard_alerts(struct lf_linecard *lp);
void fms_fma_init_xcvr_alerts(struct lf_xcvr *xcp);
void fms_fma_init_xbar_alerts(struct lf_xbar *xbp);
void fms_init_link_alerts(struct fms_link *linkp);
void fms_clear_bf_serial(struct lf_fabric *fp);
void fms_create_topo_map(struct lf_fabric *fp);
int fms_xbar_link_down(struct lf_xbar *xp, int port);
void fms_xbar_link_up(struct lf_xbar *xp, int port);
void fms_schedule_map_push(void);
void fms_cancel_map_push(void);
int fms_map_push_pending(void);
void fms_note_new_host(struct lf_host *hp);
void fms_note_changed_host(struct lf_host *hp);
void fms_commit_host(struct lf_host *hp);
void fms_flush_db_changes(void);
void fms_remove_host_from_db_fabric(struct lf_host *hp);
void fms_remove_host_from_db_table(struct lf_host *hp);
void fms_remove_host_from_db(struct lf_host *hp);
void fms_add_host_to_db_fabric(struct lf_host *hp);
void fms_add_host_to_db(struct lf_host *hp);
void fms_add_host_to_db_table(struct lf_host *hp);
void fms_update_host_in_db(struct lf_host *hp);
void fms_add_linecard_to_db(struct lf_linecard *lp);
void fms_add_linecard_to_db_fabric(struct lf_linecard *lp);
void fms_add_linecard_to_db_table(struct lf_linecard *lp);
void fms_add_enclosure_to_db(struct lf_enclosure *ep);
void fms_add_enclosure_to_db_fabric(struct lf_enclosure *ep);
void fms_add_enclosure_to_db_table(struct lf_enclosure *ep);
void fms_add_topo_link_to_db(union lf_node *np, int port);
void fms_add_topo_link_to_db_fabric(union lf_node *np, int port);
void fms_add_topo_link_to_db_table(union lf_node *np, int port);
void fms_remove_topo_link_from_db(union lf_node *np, int port);
void fms_remove_topo_link_from_db_fabric(union lf_node *np, int port);
void fms_remove_topo_link_from_db_table(union lf_node *np, int port);
void fms_remove_enclosure_from_db(struct lf_enclosure *ep);
void fms_remove_enclosure_from_db_table(struct lf_enclosure *ep);
void fms_dereference_link(struct fms_link *linkp);
void fms_note_link_removed(union lf_node *np, int port);
void fms_set_node_link_state(union lf_node *np, int port, int state);
void fms_update_topo_link_state(union lf_node *np, int port);
void fms_connect_topo_nodes(union lf_node *np1, int port1,
			    union lf_node *np2, int port2);
int fms_adjust_xbar_port_nos(struct lf_xbar *xp, int offset);
int fms_update_current_fabric_with_mapped(struct lf_fabric *fp,
                                          struct lf_fabric *rfp);
void fms_merge_new_linecard(struct lf_linecard *lp);
void fms_nic_link_up(struct lf_nic *nicp, int port);
void fms_nic_link_down(struct lf_nic *nicp, int port);
struct lf_linecard *fms_new_linecard(struct lf_enclosure *ep, int slot,
  char *product_id, char *serial_no);
int fms_db_complete(void);
int fms_linecard_maintence(struct lf_linecard *lp, int port_label, int up);
int fms_is_bogus(union lf_node *np);
